package TKL::User;

## $Id: User.pm,v 1.5 2004/12/08 15:26:25 sondberg Exp $

use TKL;
use Carp;
use strict;

our @ISA = qw(TKL);
our $mailer = "/usr/sbin/sendmail";	## This is our default MTA
my $mailer_args = "-oi -t -odq";	## These are our default arguments, remove -odq if
					## you want immediate delivery

sub user_template {
    my ($self, $record, $template) = @_;

    $template =~ s/\{(.+?)\}/$record->{$1}/g;

    return $template;
}


sub get_field {
    my ($self, $field) = @_;
    my ($node) = $self->{user_node}->findnodes($field);

    if (defined($node)) {
	return $node->tkl_cdata();
    } else {
	return undef;
    }
}


sub set_from_addr {
    my ($self, $from) = @_;

    $self->{mailer_from} = $from;
}


sub send_mail {
    my ($self, $subject, @msg) = @_;
    my $mailer = defined($self->{mailer}) ? $self->{mailer} : $mailer;
    my $mailer_args = defined($self->{mailer_args}) ? $self->{mailer_args} : $mailer_args;
    my $name = $self->get_field("name");
    my $email = $self->get_field("email");
    my $mailer_from = defined($self->{mailer_from}) ? $self->{mailer_from} : "TKL System <tkl-admin\@mydomain.com>";
    local *MAIL;

    unless (-x $mailer) {
	croak "$0: Unable to find/execute mailer '$mailer'";
    }

    open MAIL, "|$mailer $mailer_args" or croak "$0: Unable to fork mailer process: $!";
    
    print MAIL "From: $mailer_from\n";
    print MAIL "To: $name <$email>\n";
    print MAIL "Subject: $subject\n\n";
    
    foreach (@msg) {
	print MAIL "$_\n";
    }
    
    close MAIL or carp "$0: Unable to close mailer pipe nicely: $!";
}
    


1;

__END__

=head1 NAME

TKL::User - Perl package for handling user profiles and communication with users.

=head1 SYNOPSIS

  use TKL::User;

  my $tkl = new TKL::User;
  my $name = $tkl->get_field('name');
  $tkl->send_mail("Here goes the subject", "First line", "Second line");

=head1 DESCRIPTION

This perl package is a collection of methods for accessing TKL user information.
Such information is stored in users.tkl in the TKL portal root.

=head2 Methods

These are the methods currently available:

=head3 new

Constructs a new copy of a TKL::User object:

  my $user = new TKL::User(root => '/path/to/portal/root');

where the root argument is only necessary when calling scripts from outside the
portal area.

=head3 get_field

Looks up a field in users.tkl:

  my $name = $user->get_field('name');
  my $email = $user->get_field('email');

=head3 user_template

Use this method for inserting user fields into a template string. The template
should have this form

  $template = "Text bla bla bla {field1} bla bla {field2}";

and is stuffed with values this way:

  $stuffed = $user->user_template($data, $template);

where $data is a referenced hash with the following structure:

  $data = {
      		'field1'	=> 'value1',
		'field2'	=> 'value2'
  	  };

=head3 send_mail

Surprisingly enough, this method sends a mail to the user
instantiated by the object:

  $user->send_mail('subject', 'line1', 'line2', ....);

=head1 AUTHOR

Anders Snderberg Mortensen <sondberg@indexdata.dk>
Indexdata, Copenhagen.

=head1 SEE-ALSO

Man pages for various TKL packages and perl(1).

=cut
